'use strict';

const childProcess = require('child_process');

module.exports = class Util {

    static TGLOG(msg) {
        return '';
    }

    static isValidIPv4(str) {
        const blocks = str.split('.');
        if (blocks.length !== 4) return false;

        for (let value of blocks) {
            value = parseInt(value, 10);
            if (Number.isNaN(value)) return false;
            if (value < 0 || value > 255) return false;
        }

        return true;
    }

    static hasKeys(obj, keys) {
        return keys.every(key => key in obj);
    }

    static textShortner(text, length = 6) {
        if (!text || typeof text !== 'string') {
            return '';
        }
        if (text.length < (length * 2)) {
            return text;
        }
        return `${text.substring(0, length)}...${text.substring(text.length - length)}`;
    }

    static isWGConfig(text) {
        return /\[Interface\]/.test(text) && /\[Peer\]/.test(text);
    }

    static parseWireGuardConfig(configString) {
        const result = {
            Interface: {},
            Peer: {}
        };

        const excludedFields = ['Table', 'PostUp', 'PreDown'];

        const lines = configString
            .split('\n')
            .map(line => line.trim())
            .filter(line => line && !line.startsWith('ddawg_panel'));

        let currentSection = null;

        for (const line of lines) {
            if (line.startsWith('[Interface]')) {
                currentSection = 'Interface';
                continue;
            } else if (line.startsWith('[Peer]')) {
                currentSection = 'Peer';
                continue;
            }

            if (!currentSection) continue;

            const [key, value] = line.split('=').map(part => part.trim());
            if (key && value) {
                if (currentSection === 'Interface' && excludedFields.includes(key)) {
                    continue;
                }
                if (key === 'DNS' || key === 'AllowedIPs') {
                    result[currentSection][key] = value.split(',').map(v => v.trim());
                } else {
                    result[currentSection][key] = value;
                }
            }
        }
        if (result.Interface && result.Interface.PrivateKey) {
            result.Interface.shortPrivateKey = this.textShortner(result.Interface.PrivateKey, 10);
        }
        return result;
    }

    static promisify(fn) {
        // eslint-disable-next-line func-names
        return function (req, res) {
            Promise.resolve().then(async () => fn(req, res))
                .then((result) => {
                    if (res.headersSent) return;

                    if (typeof result === 'undefined') {
                        return res
                            .status(204)
                            .end();
                    }

                    return res
                        .status(200)
                        .json(result);
                })
                .catch((error) => {
                    if (typeof error === 'string') {
                        error = new Error(error);
                    }

                    // eslint-disable-next-line no-console
                    console.error(error);

                    return res
                        .status(error.statusCode || 500)
                        .json({
                            error: error.message || error.toString(),
                            stack: error.stack,
                        });
                });
        };
    }

    static async exec(cmd, {
        log = true,
    } = {}) {
        if (typeof log === 'string') {
            // eslint-disable-next-line no-console
            console.log(`$ ${log}`);
        } else if (log === true) {
            // eslint-disable-next-line no-console
            console.log(`$ ${cmd}`);
        }

        if (process.platform !== 'linux') {
            return '';
        }

        return new Promise((resolve, reject) => {
            childProcess.exec(cmd, {
                shell: 'bash',
            }, (err, stdout) => {
                if (err) return reject(err);
                return resolve(String(stdout).trim());
            });
        });
    }

};
