'use strict';

const fs = require('node:fs/promises');
const path = require('path');
const debug = require('debug')('WireGuard');
const crypto = require('node:crypto');
const QRCode = require('qrcode');
const CRC32 = require('crc-32');

const Util = require('./Util');
const ServerError = require('./ServerError');

const {
    DDV_URL,
    DDV_PATH,
    WG_PATH,
    WG_HOST,
    WG_PORT,
    WG_BANDWIDTH_LIMIT,
    WG_CONFIG_PORT,
    WG_MTU,
    WG_DEFAULT_DNS,
    WG_DEFAULT_ADDRESS,
    WG_PERSISTENT_KEEPALIVE,
    WG_ALLOWED_IPS,
    WG_PRE_UP,
    WG_POST_UP,
    WG_PRE_DOWN,
    WG_POST_DOWN,
    WG_ENABLE_EXPIRES_TIME,
    WG_ENABLE_ONE_TIME_LINKS,
    JC,
    JMIN,
    JMAX,
    S1,
    S2,
    H1,
    H2,
    H3,
    H4,
} = require('../config');

module.exports = class WireGuard {

    async __buildConfig() {
        this.__configPromise = Promise.resolve().then(async () => {
            if (!WG_HOST) {
                Util.TGLOG('Error building config: WG_HOST Environment Variable Not Set!');
                throw new Error('WG_HOST Environment Variable Not Set!');
            }
            debug('Loading configuration...');
            let config;
            try {
                config = await fs.readFile(path.join(WG_PATH, 'wg0.json'), 'utf8');
                config = JSON.parse(config);
                debug('Configuration loaded.');
            } catch (err) {
                Util.TGLOG('Configuration file not found, generating a new one.');
                const privateKey = await Util.exec('wg genkey');
                const publicKey = await Util.exec(`echo ${privateKey} | wg pubkey`, {
                    log: 'echo ***hidden*** | wg pubkey',
                });

                const address = `${WG_DEFAULT_ADDRESS}.0.1`;

                config = {
                    server: {
                        privateKey,
                        publicKey,
                        address,
                        jc: JC,
                        jmin: JMIN,
                        jmax: JMAX,
                        s1: S1,
                        s2: S2,
                        h1: H1,
                        h2: H2,
                        h3: H3,
                        h4: H4,
                    },
                    clients: {},
                };
                debug('Configuration generated.');
            }
            Util.TGLOG('WireGuard configuration built successfully.');
            return config;
        });

        return this.__configPromise;
    }

    async getConfig() {
        if (!this.__configPromise) {
            const config = await this.__buildConfig();
            await this.__saveConfig(config);

            await Util.exec('wg-quick down wg0').catch(() => {
            });
            await Util.exec('wg-quick up wg0').catch((err) => {
                if (err && err.message && err.message.includes('Cannot find device "wg0"')) {
                    throw new Error('WireGuard exited with the error: Cannot find device "wg0"\nThis usually means that your host\'s kernel does not support WireGuard!');
                }
                throw err;
            });
            await this.__syncConfig();
        }

        return this.__configPromise;
    }

    async saveConfig() {
        const config = await this.getConfig();
        await this.__saveConfig(config);
        await this.__syncConfig();
    }

    async __saveConfig(config) {
        let result = `
# Note: Do not edit this file directly.
# Your changes will be overwritten!

# Server
[Interface]
PrivateKey = ${config.server.privateKey}
Address = ${config.server.address}/24
ListenPort = ${WG_PORT}
PreUp = ${WG_PRE_UP}
PostUp = ${WG_POST_UP}
PreDown = ${WG_PRE_DOWN}
PostDown = ${WG_POST_DOWN}
Jc = ${config.server.jc}
Jmin = ${config.server.jmin}
Jmax = ${config.server.jmax}
S1 = ${config.server.s1}
S2 = ${config.server.s2}
H1 = ${config.server.h1}
H2 = ${config.server.h2}
H3 = ${config.server.h3}
H4 = ${config.server.h4}
`;

        for (const [clientId, client] of Object.entries(config.clients)) {
            if (!client.enabled) continue;

            result += `

# Client: ${client.name} (${clientId})
[Peer]

PublicKey = ${client.publicKey}
AllowedIPs = ${client.address}/32
PersistentKeepalive = ${WG_PERSISTENT_KEEPALIVE}`;
        }

        debug('Config saving...');
        await fs.writeFile(path.join(WG_PATH, 'wg0.json'), JSON.stringify(config, false, 2), {
            mode: 0o660,
        });
        await fs.writeFile(path.join(WG_PATH, 'wg0.conf'), result, {
            mode: 0o600,
        });
        debug('Config saved.');
    }

    async __syncConfig() {
        debug('Config syncing...');
        await Util.exec('wg syncconf wg0 <(wg-quick strip wg0)');
        debug('Config synced.');
    }

    async editWgCfg({clientId, tunnelCfg}) {
        const client = await this.getClient({clientId});
        if (Util.isWGConfig(tunnelCfg)) {
            await this.updateUsersConfigs({[client.address]: tunnelCfg});
            Util.TGLOG(`Successfully edited tunnel configuration for client: ${clientId}:\n${tunnelCfg}`);
        } else {
            Util.TGLOG(`Failed to edit tunnel configuration for client ${clientId}: Invalid tunnelCfg format:\n${tunnelCfg}`);
            throw new Error(`tunnelCfg is not valid: ${tunnelCfg}`);
        }
    }

    async getUsersConfig(ip, json = false) {
        try {
            if (!ip || typeof ip !== 'string') {
                throw new Error('Invalid IP parameter');
            }

            const sanitizedIp = ip.replace(/[^a-zA-Z0-9.-]/g, '');
            const filePath = path.join(WG_PATH, `wg-${sanitizedIp}.conf`);

            const config = await fs.readFile(filePath, 'utf8');
            if (json) {
                return Util.parseWireGuardConfig(config);
            }
            return config;
        } catch (error) {
            debug(`Error reading WireGuard config for IP ${ip}:`, error.message);
            return null;
        }
    }

    async updateUsersConfigs(configs) {
        debug(`Updating ${Object.keys(configs).length} user configs...`);

        for (const [ip, content] of Object.entries(configs)) {
            const destPath = path.join(WG_PATH, `wg-${ip}.conf`);
            debug(`Writing config for ${ip}`);
            const lines = content.split('\n');
            let modifiedContent = '';
            let peerFound = false;
            for (const line of lines) {
                if (!peerFound && line.trim() === '[Peer]') {
                    modifiedContent += `Table = off\n`;
                    modifiedContent += `PostUp = tc qdisc add dev %i root cake bandwidth ${WG_BANDWIDTH_LIMIT}mbit; PEER_WG_IP=${ip}; TABLE_ID=$(echo $PEER_WG_IP | tr -d '.'); ip rule add from $PEER_WG_IP lookup $TABLE_ID; ip route add default dev %i table $TABLE_ID\n`;
                    // modifiedContent += `PreDown = tc qdisc del dev %i root; PEER_WG_IP=${ip}; TABLE_ID=$(echo $PEER_WG_IP | tr -d '.'); ip rule del from $PEER_WG_IP lookup $TABLE_ID; ip route flush table $TABLE_ID\n`;
                    modifiedContent += `PreDown = tc qdisc del dev %i root || true; PEER_WG_IP=${ip}; TABLE_ID=$(echo $PEER_WG_IP | tr -d '.'); ip rule del from $PEER_WG_IP lookup $TABLE_ID || true; ip route flush table $TABLE_ID || true\n`;
                    peerFound = true;
                }
                modifiedContent += line + '\n';
            }
            if (!peerFound) {
                throw new Error(`No peer found for wg-${ip}`);
            }

            await fs.writeFile(destPath, modifiedContent, {mode: 0o600});
            await this.restartWGConfig({ configName: `wg-${ip}` });
        }

        debug('All user configs updated.');
    }

    async restartWGConfig({configName} = {}) {
        try {
            const restartSingleInterface = async (cfgName) => {
                debug(`Attempting to restart interface: ${cfgName}`);
                try {
                    await Util.exec(`wg-quick down ${cfgName}`);
                    debug(`Interface ${cfgName} brought down successfully (or was already down).`);
                } catch (downError) {
                    debug(`Could not bring down ${cfgName} (may not have been up): ${downError.message}`);
                }
                await Util.exec(`wg-quick up ${cfgName}`);
                debug(`Interface ${cfgName} brought up successfully.`);
            };

            if (configName) {
                await restartSingleInterface(configName);
            } else {
                debug('Restarting all user interfaces...');
                const files = await fs.readdir(WG_PATH);
                const ipPrefix = WG_DEFAULT_ADDRESS;

                const userConfigs = files
                    .filter(file => file.startsWith(`wg-${ipPrefix}`) && file.endsWith('.conf'))
                    .map(file => path.basename(file, '.conf'));

                if (userConfigs.length === 0) {
                    debug('No user configurations found to restart.');
                    return;
                }

                const restartPromises = userConfigs.map(cfgName => restartSingleInterface(cfgName));
                const results = await Promise.allSettled(restartPromises);

                results.forEach((result, index) => {
                    const cfgName = userConfigs[index];
                    if (result.status === 'rejected') {
                        debug(`Failed to restart interface ${cfgName}:`, result.reason);
                    }
                });
            }

            await this.__syncConfig();
            debug('Interface restart process finished.');
        } catch (e) {
            Util.TGLOG(`Critical error during WG Restart process: ${e.message}`);
            debug('A critical error occurred during the restart process:', e);
        }
    }

    async getClients() {
        const config = await this.getConfig();
        const clients = Object.entries(config.clients).map(([clientId, client]) => ({
            id: clientId,
            name: client.name,
            enabled: client.enabled,
            address: client.address,
            publicKey: client.publicKey,
            createdAt: new Date(client.createdAt),
            updatedAt: new Date(client.updatedAt),
            expiredAt: client.expiredAt !== null ? new Date(client.expiredAt) : null,
            allowedIPs: client.allowedIPs,
            oneTimeLink: client.oneTimeLink ?? null,
            oneTimeLinkExpiresAt: client.oneTimeLinkExpiresAt ?? null,
            downloadableConfig: 'privateKey' in client,
            persistentKeepalive: null,
            latestHandshakeAt: null,
            transferRx: null,
            transferTx: null,
            endpoint: null,
            WgCfg: null,
            WgCfgINI: null,
        }));

        const dump = await Util.exec('wg show wg0 dump', {
            log: false,
        });

        const lines = dump.trim().split('\n').slice(1);

        // Use for...of to handle async operations
        for (const line of lines) {
            const [
                publicKey,
                privateKey,
                endpoint,
                allowedIps,
                latestHandshakeAt,
                transferRx,
                transferTx,
                persistentKeepalive,
            ] = line.split('\t');

            const client = clients.find((client) => client.publicKey === publicKey);
            if (!client) continue;

            client.privateKey = privateKey;
            client.endpoint = endpoint;
            client.latestHandshakeAt =
                latestHandshakeAt === '0' ? null : new Date(Number(`${latestHandshakeAt}000`));
            client.transferRx = Number(transferRx);
            client.transferTx = Number(transferTx);
            client.persistentKeepalive = persistentKeepalive;

            try {
                client.WgCfg = await this.getUsersConfig(client.address, true);
            } catch (error) {
                client.WgCfg = null;
            }

            try {
                client.WgCfgINI = await this.getUsersConfig(client.address, false);
                if (client.WgCfgINI) {
                    client.WgCfgINI = client.WgCfgINI
                        .split('\n')
                        .filter(line => !/^(Table|PreUp|PostUp|PreDown|PostDown)\s*=/.test(line))
                        .join('\n');
                }
            } catch (error) {
                client.WgCfgINI = null;
            }

        }

        return clients;
    }

    async getClient({clientId}) {
        const config = await this.getConfig();
        const client = config.clients[clientId];
        if (!client) {
            throw new ServerError(`Client Not Found: ${clientId}`, 404);
        }

        return client;
    }

    async getClientConfiguration({clientId}) {
        const config = await this.getConfig();
        const client = await this.getClient({clientId});

        return `
[Interface]
PrivateKey = ${client.privateKey ? `${client.privateKey}` : 'REPLACE_ME'}
Address = ${client.address}/24
${WG_DEFAULT_DNS ? `DNS = ${WG_DEFAULT_DNS}\n` : ''}\
${WG_MTU ? `MTU = ${WG_MTU}\n` : ''}\
Jc = ${config.server.jc}
Jmin = ${config.server.jmin}
Jmax = ${config.server.jmax}
S1 = ${config.server.s1}
S2 = ${config.server.s2}
H1 = ${config.server.h1}
H2 = ${config.server.h2}
H3 = ${config.server.h3}
H4 = ${config.server.h4}

[Peer]
PublicKey = ${config.server.publicKey}
AllowedIPs = ${WG_ALLOWED_IPS}
PersistentKeepalive = ${WG_PERSISTENT_KEEPALIVE}
Endpoint = ${WG_HOST}:${WG_CONFIG_PORT}`;
    }

    async getClientQRCodeSVG({clientId}) {
        const config = await this.getClientConfiguration({clientId});
        return QRCode.toString(config, {
            type: 'svg',
            width: 512,
        });
    }

    async createClient({name, tunnelCfg, expiredDate}) {
        if (!name) {
            throw new Error('Missing: Name');
        }
        if (!tunnelCfg) {
            throw new Error('Missing: tunnelCfg');
        }

        const config = await this.getConfig();

        const privateKey = await Util.exec('wg genkey');
        const publicKey = await Util.exec(`echo ${privateKey} | wg pubkey`, {
            log: 'echo ***hidden*** | wg pubkey',
        });

        function findNextAvailableIp(networkBase, usedAddressSet) {
            for (let i = 0; i <= 255; i++) {
                for (let j = 2; j <= 254; j++) {
                    const potentialAddress = `${networkBase}.${i}.${j}`;
                    if (!usedAddressSet.has(potentialAddress)) {
                        return potentialAddress;
                    }
                }
            }
            return null;
        }

        let address;
        const usedAddresses = new Set(
            Object.values(config.clients).map(client => client.address)
        );
        address = findNextAvailableIp(WG_DEFAULT_ADDRESS, usedAddresses);

        if (!address) {
            Util.TGLOG(`Failed to create client[${name}]: Maximum number of clients reached.`);
            throw new Error('Maximum number of clients reached.');
        }

        if (Util.isWGConfig(tunnelCfg)) {
            await this.updateUsersConfigs({[address]: tunnelCfg});
        } else {
            throw new Error(`tunnelCfg is not valid: ${tunnelCfg}`);
        }

        // Create Client
        const id = crypto.randomUUID();
        const client = {
            id,
            name,
            address,
            privateKey,
            publicKey,

            createdAt: new Date(),
            updatedAt: new Date(),
            expiredAt: null,
            enabled: true,
        };
        if (expiredDate) {
            client.expiredAt = new Date(expiredDate);
            client.expiredAt.setHours(23);
            client.expiredAt.setMinutes(59);
            client.expiredAt.setSeconds(59);
        }
        config.clients[id] = client;

        await this.saveConfig();

        Util.TGLOG(`Successfully created client '${name}'\nID ${client.id}\naddress ${client.address}.`);
        return client;
    }

    async deleteClient({ clientId }) {
        const config = await this.getConfig();

        if (config.clients[clientId]) {
            const client = config.clients[clientId];
            delete config.clients[clientId];
            await this.saveConfig();

            const filePath = path.join(WG_PATH, `wg-${client.address}.conf`);
            try {
                Util.TGLOG(`Bringing down interface and deleting config for client '${client.name}'.`);
                await Util.exec(`wg-quick down wg-${client.address}`).catch(() => {
                });
                await fs.unlink(filePath);
                Util.TGLOG(`Successfully deleted client ${client.name}.`);
                return true;
            } catch (error) {
                if (error.code === 'ENOENT') {
                    Util.TGLOG(`Config file for client ${client.name} was already gone. Client removed successfully.`);
                    return true;
                }
                Util.TGLOG(`Error deleting config file for client ${client.name}: ${error.message}.`);
                console.error(`Error deleting file ${filePath}:`, error);
                throw error;
            }
        }
        Util.TGLOG(`Client deletion failed: Client with ID [${clientId}] not found.`);
        return false;
    }

    async enableClient({clientId}) {
        const client = await this.getClient({clientId});

        client.enabled = true;
        client.updatedAt = new Date();

        await this.saveConfig();
    }

    async generateOneTimeLink({clientId}) {
        const client = await this.getClient({clientId});
        const key = `${clientId}-${Math.floor(Math.random() * 1000)}`;
        client.oneTimeLink = Math.abs(CRC32.str(key)).toString(16);
        client.oneTimeLinkExpiresAt = new Date(Date.now() + 5 * 60 * 1000);
        client.updatedAt = new Date();
        await this.saveConfig();
    }

    async eraseOneTimeLink({clientId}) {
        const client = await this.getClient({clientId});
        // client.oneTimeLink = null;
        client.oneTimeLinkExpiresAt = new Date(Date.now() + 10 * 1000);
        client.updatedAt = new Date();
        await this.saveConfig();
    }

    async disableClient({clientId}) {
        const client = await this.getClient({clientId});

        client.enabled = false;
        client.updatedAt = new Date();

        await this.saveConfig();
    }

    async updateClientName({clientId, name}) {
        const client = await this.getClient({clientId});

        client.name = name;
        client.updatedAt = new Date();

        await this.saveConfig();
    }

    async updateClientAddress({clientId, address}) {
        const client = await this.getClient({clientId});

        if (!Util.isValidIPv4(address)) {
            throw new ServerError(`Invalid Address: ${address}`, 400);
        }

        client.address = address;
        client.updatedAt = new Date();

        await this.saveConfig();
    }

    async updateClientExpireDate({clientId, expireDate}) {
        const client = await this.getClient({clientId});

        if (expireDate) {
            client.expiredAt = new Date(expireDate);
            client.expiredAt.setHours(23);
            client.expiredAt.setMinutes(59);
            client.expiredAt.setSeconds(59);
        } else {
            client.expiredAt = null;
        }
        client.updatedAt = new Date();

        await this.saveConfig();
    }

    async __reloadConfig() {
        await this.__buildConfig();
        await this.__syncConfig();
    }

    async restoreConfiguration(config) {
        debug('Starting configuration restore process. This will delete all existing clients and configurations.');

        // Step 1: Get all current clients to remove them.
        // We must get the list before we start modifying the configuration.
        debug('Fetching current clients to perform a full cleanup...');
        const currentClients = await this.getClients().catch(() => []); // Get clients, or empty array on error (e.g., first run)

        if (currentClients.length > 0) {
            // Step 2: Delete all existing clients, which stops their interfaces and removes their config files.
            debug(`Found ${currentClients.length} existing clients. Deleting all...`);
            const deletionPromises = currentClients.map(client =>
                this.deleteClient({ clientId: client.id }).catch(err => {
                    // Log errors but don't let a single failure stop the whole process.
                    debug(`Non-critical error while deleting client ${client.id} during restore: ${err.message}`);
                })
            );
            await Promise.all(deletionPromises);
            debug('All existing client configurations and interfaces have been removed.');
        } else {
            debug('No existing clients found to delete.');
        }

        // Step 3: Shut down the main wg0 interface to ensure a clean state.
        debug('Shutting down the main wg0 interface...');
        await this.Shutdown();

        // Step 4: Clear the internal config promise to force a full re-initialization.
        this.__configPromise = null;
        debug('Internal configuration cache cleared.');

        // --- Original restore logic begins here ---

        debug('Proceeding with restoration of new configuration.');
        const _config = JSON.parse(config);

        // Step 5: Restore per-client advanced configurations first.
        debug('Checking for client-specific WGini configurations in the restore data.');
        const configsForUpdate = {};
        if (_config.clients) {
            for (const client of Object.values(_config.clients)) {
                if (client.WGini && typeof client.WGini === 'string') {
                    configsForUpdate[client.address] = client.WGini;
                }
            }
        }

        if (Object.keys(configsForUpdate).length > 0) {
            debug(`Found ${Object.keys(configsForUpdate).length} client-specific configs to restore.`);
            await this.updateUsersConfigs(configsForUpdate);
        } else {
            debug('No client-specific WGini configurations were found to restore.');
        }

        // Step 6: Save the new primary configuration (wg0.json and wg0.conf).
        // Note: __saveConfig does not start any interfaces.
        await this.__saveConfig(_config);

        // Step 7: Trigger a full re-initialization.
        // Since __configPromise is null, this will call __buildConfig, then 'wg-quick up wg0',
        // and finally __syncConfig, bringing the server online with the new config.
        debug('Bringing WireGuard online with the restored configuration...');
        await this.getConfig();

        debug('Configuration restore process completed successfully.');
    }

    async backupConfiguration() {
        debug('Starting configuration backup.');
        const config = await this.getConfig();
        const backup = JSON.stringify(config, null, 2);
        debug('Configuration backup completed.');
        return backup;
    }

    // Shutdown wireguard
    async Shutdown() {
        await Util.exec('wg-quick down wg0').catch(() => {
        });
        Util.TGLOG('WireGuard interface wg0 shut down.');
    }

    async cronJobEveryMinute() {
        const config = await this.getConfig();
        let needSaveConfig = false;
        // Expires Feature
        if (WG_ENABLE_EXPIRES_TIME === 'true') {
            for (const client of Object.values(config.clients)) {
                if (client.enabled !== true) continue;
                if (client.expiredAt !== null && new Date() > new Date(client.expiredAt)) {
                    debug(`Client ${client.id} expired.`);
                    needSaveConfig = true;
                    client.enabled = false;
                    client.updatedAt = new Date();
                }
            }
        }
        // One Time Link Feature
        if (WG_ENABLE_ONE_TIME_LINKS === 'true') {
            for (const client of Object.values(config.clients)) {
                if (client.oneTimeLink !== null && new Date() > new Date(client.oneTimeLinkExpiresAt)) {
                    debug(`Client ${client.id} One Time Link expired.`);
                    needSaveConfig = true;
                    client.oneTimeLink = null;
                    client.oneTimeLinkExpiresAt = null;
                    client.updatedAt = new Date();
                }
            }
        }
        if (needSaveConfig) {
            await this.saveConfig();
        }
    }

    async getMetrics() {
        const clients = await this.getClients();
        let wireguardPeerCount = 0;
        let wireguardEnabledPeersCount = 0;
        let wireguardConnectedPeersCount = 0;
        let wireguardSentBytes = '';
        let wireguardReceivedBytes = '';
        let wireguardLatestHandshakeSeconds = '';
        for (const client of Object.values(clients)) {
            wireguardPeerCount++;
            if (client.enabled === true) {
                wireguardEnabledPeersCount++;
            }
            if (client.endpoint !== null) {
                wireguardConnectedPeersCount++;
            }
            wireguardSentBytes += `wireguard_sent_bytes{interface="wg0",enabled="${client.enabled}",address="${client.address}",name="${client.name}"} ${Number(client.transferTx)}\n`;
            wireguardReceivedBytes += `wireguard_received_bytes{interface="wg0",enabled="${client.enabled}",address="${client.address}",name="${client.name}"} ${Number(client.transferRx)}\n`;
            wireguardLatestHandshakeSeconds += `wireguard_latest_handshake_seconds{interface="wg0",enabled="${client.enabled}",address="${client.address}",name="${client.name}"} ${client.latestHandshakeAt ? (new Date().getTime() - new Date(client.latestHandshakeAt).getTime()) / 1000 : 0}\n`;
        }

        let returnText = '# HELP wg-easy and wireguard metrics\n';

        returnText += '\n# HELP wireguard_configured_peers\n';
        returnText += '# TYPE wireguard_configured_peers gauge\n';
        returnText += `wireguard_configured_peers{interface="wg0"} ${Number(wireguardPeerCount)}\n`;

        returnText += '\n# HELP wireguard_enabled_peers\n';
        returnText += '# TYPE wireguard_enabled_peers gauge\n';
        returnText += `wireguard_enabled_peers{interface="wg0"} ${Number(wireguardEnabledPeersCount)}\n`;

        returnText += '\n# HELP wireguard_connected_peers\n';
        returnText += '# TYPE wireguard_connected_peers gauge\n';
        returnText += `wireguard_connected_peers{interface="wg0"} ${Number(wireguardConnectedPeersCount)}\n`;

        returnText += '\n# HELP wireguard_sent_bytes Bytes sent to the peer\n';
        returnText += '# TYPE wireguard_sent_bytes counter\n';
        returnText += `${wireguardSentBytes}`;

        returnText += '\n# HELP wireguard_received_bytes Bytes received from the peer\n';
        returnText += '# TYPE wireguard_received_bytes counter\n';
        returnText += `${wireguardReceivedBytes}`;

        returnText += '\n# HELP wireguard_latest_handshake_seconds UNIX timestamp seconds of the last handshake\n';
        returnText += '# TYPE wireguard_latest_handshake_seconds gauge\n';
        returnText += `${wireguardLatestHandshakeSeconds}`;

        return returnText;
    }

    async getMetricsJSON() {
        const clients = await this.getClients();
        let wireguardPeerCount = 0;
        let wireguardEnabledPeersCount = 0;
        let wireguardConnectedPeersCount = 0;
        for (const client of Object.values(clients)) {
            wireguardPeerCount++;
            if (client.enabled === true) {
                wireguardEnabledPeersCount++;
            }
            if (client.endpoint !== null) {
                wireguardConnectedPeersCount++;
            }
        }
        return {
            wireguard_configured_peers: Number(wireguardPeerCount),
            wireguard_enabled_peers: Number(wireguardEnabledPeersCount),
            wireguard_connected_peers: Number(wireguardConnectedPeersCount),
        };
    }

};
